import React from "react";
import { AuthLayout } from "../layout/AuthLayout";
import { AppContext } from "../../appContext";
import Switch from "../core/form/Switch";
import mapboxgl from "mapbox-gl";
import "./map.css";
import TimePicker from "../core/elements/TimePicker";
import { TransitionGroup, CSSTransition } from "react-transition-group";
import Select from "react-select";

class Map extends React.Component {
    static contextType = AppContext;
    constructor(props) {
        super(props);
        this.state = {
            address: {
                province: 11,
                city: 1,
                region: 14,
                parish: 1829,
            },
            location: {},
            zoom: 6,
            lat: 32.57,
            lng: 52.67,
            loading: true,
            reserve: false,
        };
        this.mapContainer = React.createRef();

        try {
            mapboxgl.setRTLTextPlugin(
                "https://cdn.parsimap.ir/third-party/mapbox-gl-js/plugins/mapbox-gl-rtl-text/v0.2.3/mapbox-gl-rtl-text.js",
                null
            );
        } catch (error) {}

        mapboxgl.accessToken =
            "pk.eyJ1IjoidG9vZmFubXgiLCJhIjoiY2psbmtvMHNwMWZheTNzcGlsZWRmamZrMSJ9.Zbu1kS2TaAAeRph6QuCKqQ";
        this.reserveToggleHandle = this.reserveToggleHandle.bind(this);
        this.handleProvinceChange = this.handleProvinceChange.bind(this);
        this.handleCityChange = this.handleCityChange.bind(this);
        this.handleRegionChange = this.handleRegionChange.bind(this);
        this.handleParishChange = this.handleParishChange.bind(this);
        this.updateLocationData = this.updateLocationData.bind(this);
        this.handleTextChange = this.handleTextChange.bind(this);
    }

    componentDidMount() {
        this.getAddress();
        this.getLocation();
        const map = new mapboxgl.Map({
            container: this.mapContainer,
            style: "https://www.parsimap.ir/styles/ggl.json",
            center: [this.state.lng, this.state.lat],
            zoom: this.state.zoom,
        });

        map.on("move", () => {
            this.setState({
                lng: map.getCenter().lng.toFixed(4),
                lat: map.getCenter().lat.toFixed(4),
                zoom: map.getZoom().toFixed(2),
            });
        });
    }

    async getLocation() {
        try {
            const response = await axios.post("/request/getLocation");
            this.setState({ loading: false, location: response.data });
        } catch (error) {
            this.setState({ loading: false });
        }
        console.log(this.state.location.cityList);
    }

    handleSubmitClick() {
        this.props.history.push("/products");
    }

    async getAddress() {
        this.setState({ loading: true });

        try {
            const response = await axios.post("/request/getUserAddress");
            this.setState({ loading: false, address: response.data });
        } catch (error) {
            this.setState({ loading: false });
        }
    }

    reserveToggleHandle() {
        this.setState((prevState) => ({
            reserve: !prevState.reserve,
        }));
    }

    currentLocation() {}

    handleProvinceChange(newValue) {
        this.setState((prevState) => ({
            address: {
                ...prevState.address,
                province: newValue.value,
            },
        }));
        this.updateLocationData("city", newValue.value);
    }

    handleCityChange(newValue) {
        this.setState((prevState) => ({
            address: {
                ...prevState.address,
                city: newValue.value,
            },
        }));
        this.updateLocationData("region", newValue.value);
    }

    handleRegionChange(newValue) {
        this.setState((prevState) => ({
            address: {
                ...prevState.address,
                region: newValue.value,
            },
        }));
        this.updateLocationData("parish", newValue.value);
    }

    handleParishChange(newValue) {
        this.setState((prevState) => ({
            address: {
                ...prevState.address,
                parish: newValue.value,
            },
        }));
    }

    async getLocationTypeAjax(type, id) {
        this.setState({ loading: true });
        if (!this.state.loading) {
            try {
                const response = await axios.post(
                    "/request/get" +
                        type.replace(/\b\w/g, (l) => l.toUpperCase()),
                    { id: id }
                );

                const myKey =
                    type.replace(/\b\w/g, (l) => l.toLowerCase()) + "List";
                this.setState((prevState) => {
                    let location = prevState.location[myKey];
                    location[id] = response.data;
                    return {
                        location: {
                            ...prevState.location,
                            [myKey]: location,
                        },
                    };
                });

                this.setState({ loading: false });
            } catch (error) {
                this.setState({ loading: false });
            }
        }
    }
    updateLocationData(type, id) {
        console.log(type, id);

        const myKey = type.replace(/\b\w/g, (l) => l.toLowerCase()) + "List";
        const list = this.state.location[myKey];

        if (list && Object.values(list).length > 0) {
            if (!(id in list)) {
                this.getLocationTypeAjax(type, id);
            }
        } else {
            return [];
        }
    }

    handleTextChange(event){
        let nam = event.target.name;
        let val = event.target.value;
        this.setState({[nam]: val});
    }

    render() {
        const address = this.state.address;
        const logoSrc = "public/eqlid/logo-white.webp";
        const provinceList = this.state.location.provinceList ?? {};
        const cityList = this.state.location.cityList ?? {};
        const regionList = this.state.location.regionList ?? {};
        const parishList = this.state.location.parishList ?? {};
        const customStyles = {
            container: (provided) => ({
                ...provided,
                marginTop: "8px",
            }),
            control: (provided) => ({
                ...provided,
                border: 0,
                backgroundColor: "#f7f7f7",
            }),
            option: (provided, state) => ({
                ...provided,
                color: state.isSelected
                    ? "var(--color-secondary)"
                    : "var(--color-text)",
                backgroundColor: state.isSelected ? "var(--color-primary)" : "",
                padding: "4px 12px",
            }),
            indicatorContainer: (provided) => ({
                ...provided,
                padding: "2px",
            }),
        };
        const top = (
            <>
                <div
                    ref={(el) => (this.mapContainer = el)}
                    className="mapContainer"
                />
                <div className="navigatePanel">
                    <div className="addressRow">
                        <button
                            className="circleBtn mapBtn"
                            tabIndex="0"
                            type="button"
                            aria-label="favorite"
                            onClick={this.currentLocation}
                        >
                            <svg
                                className="svgIcon"
                                xmlns="http://www.w3.org/2000/svg"
                                focusable="false"
                                viewBox="0 0 24 24"
                                aria-hidden="true"
                            >
                                <path d="M12.545 22.418L9.2 14.47 1 10.763l.954-2.038L19.495 3 21 4.544 14.72 21.95l-2.175.468zM2.99 10.027l7.356 3.326 3.093 7.348.16-.034 5.683-15.751-.206-.211L3.03 9.94l-.04.086z" />
                            </svg>
                        </button>
                        <div className="addressItems">
                            
                            <button
                                type="submit"
                                className="button primary-box addressBtn"
                            >
                                پیش فرض
                            </button>
                            <button
                                type="submit"
                                className="button primary-box addressBtn"
                            >
                                شرکت
                            </button>
                        </div>
                    </div>
                    <div className="headerRow">
                        <button
                            className="circleBtn mapBtn"
                            tabIndex="0"
                            type="button"
                            aria-label="favorite"
                        >
                            <svg
                                className="svgIcon"
                                focusable="false"
                                viewBox="0 0 500 500"
                                aria-hidden="true"
                            >
                                <path d="M300 450c-6.4 0-12.8-2.4-17.7-7.3-9.8-9.8-9.8-25.6 0-35.4L414.6 275H25c-13.8 0-25-11.2-25-25s11.2-25 25-25h389.6L282.3 92.7c-9.8-9.8-9.8-25.6 0-35.4s25.6-9.8 35.4 0l175 175c2.4 2.4 4.2 5.2 5.4 8.2 1.2 2.9 1.9 6.1 1.9 9.4v.1c0 3.3-.7 6.5-1.9 9.4-1.2 3-3 5.7-5.4 8.2l-175 175c-4.9 5-11.3 7.4-17.7 7.4z"></path>
                            </svg>
                        </button>
                    </div>
                    <div className="centerRow">
                        <img
                            src="/public/eqlid/origin.png"
                            width="50px"
                            alt="origin marker"
                        />
                    </div>
                </div>
            </>
        );
        return (
            <AuthLayout className="mapBox" top={top} {...this.props}>
                <div>
                    <div className="row no-gutters">
                        <div className="col-12">
                            <h4 className="headerText">
                                مشخصات آدرس را وارد نمائید.
                            </h4>
                        </div>
                        
                        <div className="col-6 pl-10px">
                            <Select
                                value={
                                    Object.values(provinceList).length > 0 &&
                                    provinceList.filter((option) => {
                                        return (
                                            option.value ==
                                            this.state.address.province
                                        );
                                    })
                                }
                                options={provinceList}
                                placeholder="استان"
                                onChange={this.handleProvinceChange}
                                isRtl={true}
                                styles={customStyles}
                            />
                        </div>
                        <div className="col-6">
                            <Select
                                value={
                                    Object.values(cityList).length > 0 &&
                                    (
                                        cityList[this.state.address.province] ??
                                        []
                                    ).filter((option) => {
                                        return (
                                            option.value ==
                                            this.state.address.city
                                        );
                                    })
                                }
                                options={
                                    cityList[this.state.address.province] ?? []
                                }
                                placeholder="شهر"
                                onChange={this.handleCityChange}
                                isRtl={true}
                                styles={customStyles}
                            />
                        </div>
                        <div className="col-6 pl-10px">
                            <Select
                                value={
                                    Object.values(regionList).length > 0 &&
                                    (regionList[this.state.address.city]??[]).filter(
                                        (option) => {
                                            return (
                                                option.value ==
                                                this.state.address.region
                                            );
                                        }
                                    )
                                }
                                options={regionList[this.state.address.city]}
                                placeholder="منطقه"
                                onChange={this.handleRegionChange}
                                isRtl={true}
                                styles={customStyles}
                                menuPlacement="auto"
                            />
                        </div>
                        <div className="col-6">
                            <Select
                                value={
                                    Object.values(parishList).length > 0 &&
                                    (parishList[this.state.address.region] ?? []).filter((option) => {
                                        return (
                                            option.value ==
                                            this.state.address.parish
                                        );
                                    })
                                }
                                options={parishList[this.state.address.region]}
                                placeholder="محله"
                                onChange={this.handleParishChange}
                                isRtl={true}
                                styles={customStyles}
                                menuPlacement="auto"
                            />
                        </div>
                        <div className="col-12">
                            <textarea
                                type="text"
                                className="addressTextare"
                                placeholder="آدرس دقیق و توضیحات (پلاک، واحد و …)"
                                maxLength="1000"
                            ></textarea>
                        </div>
                        <div className="col-12">
                            <div className="reservationItem">
                                <Switch
                                    theme="default"
                                    className="qi-container"
                                    label="رزرو درخواست برای آینده"
                                    enabled={false}
                                    onStateChanged={this.reserveToggleHandle}
                                />
                                <TransitionGroup component={null}>
                                    {this.state.reserve && (
                                        <CSSTransition
                                            classNames="animation"
                                            timeout={300}
                                        >
                                            <div>
                                                <TimePicker></TimePicker>
                                            </div>
                                        </CSSTransition>
                                    )}
                                </TransitionGroup>
                            </div>
                        </div>
                    </div>
                </div>
            </AuthLayout>
        );
    }
}

export default Map;
