import React from "react";
import { Link } from "react-router-dom";
import { AuthLayout } from "../layout/AuthLayout";
import { AppContext } from "../../appContext";
import * as UrlUtils from '../../utils/Url';
import CodeInput from '../core/elements/CodeInput';
import CountDown from "../core/elements/CountDown";

export class SignIn extends React.Component {
    static contextType = AppContext;
    constructor(props) {
        super(props);
        
        this.state = {
            identified: false,
            loading: false,
            errors: {mobile:null},
            mobile: '',
            message: {},
            timerCompleted: false,
        };
        this.signIn = this.signIn.bind(this);
        this.handleChangeMobileInput = this.handleChangeMobileInput.bind(this);
        this.nextLevelSubmit = this.nextLevelSubmit.bind(this);
        this.completeCountDownHandle = this.completeCountDownHandle.bind(this);
        this.reIdentified = this.reIdentified.bind(this);
        this.codeSubmit = this.codeSubmit.bind(this);
    }

    //cycle
    componentDidMount() {
        
        if(this.context)
            this.state.authenticate = this.context.authenticate;

        if (this.state.identified) {
            this.props.history.push('/');
        }

        const { location } = this.props;

        const q = UrlUtils.queryParams(location.search);

        if (q.hasOwnProperty('mobile') && q.mobile !== '') {
            this.identify(q.mobile, {});
        }
    }

    componentDidUpdate() {
        this.state.authenticate = this.context.authenticate;
    }

    componentWillUnmount() {
        this.state.authenticate = this.context.authenticate;
    }

    async identify  (mobile = null, { setErrors }){
        this.setState({loading:true});
        try {
            
            const response = await axios.post("/user/sendOtp", {
                mobile
            });

            const { history, location } = this.props;
            if(response.data.success){
                this.setState({identified: true, loading:false});
            }
            
            const queryString = UrlUtils.queryString({
                mobile
            });

            if (queryString === location.search) {
                return;
            }

            history.push(`${location.pathname}${queryString}`);
        } catch (error) {
            if (!error.response) {
                throw new Error('Unknown error');
            }

            const { errors } = error.response.data;

            if (errors) {
                this.setState(prevState => ({
                    errors: {
                        ...prevState.errors,
                        authenticate: errors
                    }
                }))
            }

            this.setState({loading:false});
        }
    };

    /**
     * This should send an API request to request for authentication.
     * Reload if authenticated.
     *
     * @param {object} values
     * @param {object} form
     *
     * @return {undefined}
     */
    async signIn(values, form = {}) {
        this.setState({ loading: true });

        try {
            const { code , mobile } = values;

            const response = await axios.post("/user/verifyOtp2", {
                code,
                mobile
            });
            console.log('signIn',response);
            this.state.authenticate(response.data.user);
            this.setState({ loading: false });
            this.props.history.push('/');
        } catch (error) {
            if (!error.response) {
                throw new Error("Unknown error");
            }

            const { errors } = error.response.data;

            if (errors) {
                this.setState(prevState => ({
                    errors: {
                        ...prevState.errors,
                        authenticate: errors
                    }
                }))
            }

            this.setState({ loading: false });
        }
    }

     /**
     * Event listener that is called on usernameChip clicked.
     *
     * @return {undefined}
     */
    handleMobileChangeClicked = (e) => {
        e.preventDefault();
        this.setState({mobile:'', identified: false, timerCompleted: false});
        const { history, location } = this.props;
        history.push(`${location.pathname}`);
    };

    handleChangeMobileInput (event)  {
        this.setState({mobile: event.target.value});
        if(this.state.errors.mobile){
            this.setState(prevState => ({
                errors: {
                    ...prevState.errors,
                    mobile: null
                }
            }))
        }
    }

    async nextLevelSubmit  (event) {
        event.preventDefault();
        this.setState({timerCompleted: false});
        if(this.state.mobile.length != 11){
            this.setState(prevState => ({
                errors: {
                    ...prevState.errors,
                    mobile: 'شماره موبایل واردشده اشتباه است.'
                }
            }))
            return false;
        }
        
        try {
            await this.identify(this.state.mobile, event.target);
        } catch (error) {
            this.setState({
                loading: false,
                message: {
                    type: "error",
                    title: "Something went wrong",
                    body: "Oops? Something went wrong here. Please try again.",
                    action: () => window.location.reload()
                }
            });
        }

    }

    async codeSubmit(code){
        
        const values = {
            mobile: this.state.mobile,
            code: code
        };
        try {
            await this.signIn(values, event.target);
        } catch (error) {
            this.setState({
                loading: false,
                message: {
                    type: "error",
                    title: "Something went wrong",
                    body: "Oops? Something went wrong here. Please try again.",
                    action: () => window.location.reload()
                }
            });
        }
    }

    escapeSubmit (event)  {
        event.preventDefault();
    }

    completeCountDownHandle(){
        this.setState({timerCompleted: true});
    }

    reIdentified(event){
        this.setState({timerCompleted: false});
        this.nextLevelSubmit(event);
    }


    render() {
        this.state.authenticate = this.context;
        const logoSrc = "public/eqlid/logo-white.webp";
        const top =  <img src={logoSrc} className="logo-img"></img>;
        const mobileError = <div className="invalid-cellphone font-xs error">
            <i className="icon-0-2-29 icon-d14-0-2-130 "></i>
            <span >{this.state.errors.mobile}</span>
            </div>;
            
        return (
            <AuthLayout className="form-signin" top={!this.state.identified ? top : null}>
                {!this.state.identified ? (
                    <>
                <div className="main">
                    <div>
                        <p className="bold font-xl">
                            به خانواده گلدیران خوش آمدید.
                        </p>
                        <p className="mt-xl">
                            جهت استفاده بهینه از خدمات گلدیران می توانید از همان
                            ابتدا با شماره خود وارد سیستم شوید
                        </p>
                    </div>
                    <div className="form-controller mt-xxl">
                        <div className="label bold">شماره موبایل</div>
                        <div className="text-input mt-l">
                            <div className="input-container">
                                <input
                                    type="tel"
                                    className="input"
                                    placeholder="مثلاً ۰۹۱۲۳۴۵۶۷۸۹"
                                    autoCorrect="off"
                                    id="login-input"
                                    maxLength="11"
                                    value={this.state.mobile}
                                    onChange={this.handleChangeMobileInput}
                                />
                            </div>
                        </div>
                    </div>
                    {this.state.errors.mobile && mobileError}
                </div>
                <footer className="mt-l">
                    <button
                        type="submit"
                        className="button dash-button"
                        onClick={this.handleEscape}
                    >
                        شماره موبایل نمی دهم!
                    </button>
                    <button
                        type="submit"
                        className="button primary-box"
                        onClick={this.nextLevelSubmit}
                    >
                        مرحله بعد
                    </button>
                </footer>
                </>
                ) : (
                    <>
                    <header>
                        <a onClick={this.handleMobileChangeClicked}>
                            <i className="icon-arrow-right"></i>
                        </a>
                    </header>
                    <div className="main">
                        <div>
                            <p className="bold font-xl">
                            کد فعال‌سازی را وارد کنید.
                            </p>
                            <p className="mt-xl">
                                 کد 4 رقمی ارسال شده به شماره 
                                <span> {this.state.mobile} </span>
                                 را وارد نمائید. 
                                <a className="color-primary d-block"  onClick={this.handleMobileChangeClicked}>
                                    <span>تغییر شماره موبایل</span>
                                </a>
                            </p>
                            
                            <CodeInput
                                className = "mt-xl"
                                onComplete={this.codeSubmit}
                                />
                            
                        </div>
                    </div>
                    <footer className="mt-l">
                        <div >
                            پیامکی دریافت نکرده اید ؟
                        </div>
                        {this.state.timerCompleted ? (
                            <>
                            <a className="color-primary" onClick={this.reIdentified}>
                                ارسال مجدد کد
                            </a>
                            </>
                        ) : (
                            <>
                            <div>
                                <CountDown onComplete={this.completeCountDownHandle} restart={!this.state.timerCompleted} format="ss"/>
                                ثانیه تا ارسال مجدد کد
                            </div>
                            </>
                        )
                        }
                        
                    </footer>
                    </>
                )}
            </AuthLayout>
        );
    }
}

