<?php

namespace Modules\Core\Entities\Fields;

use Modules\Core\Exceptions\Form\FieldValidationException;
use Illuminate\Support\Facades\Validator;

abstract class Field
{
    /**
     * @var string
     */
    public $name;

    /**
     * @var string
     */
    protected $label;

    /**
     * @var string
     */
    protected $helpMessage;

    /**
     * @var bool
     */
    protected $readOnly;

    /**
     * @var meta
     */
    protected $withMeta;

    /**
     * @var field
     */
    protected $field;

    /**
     * @param string $key
     * @param string $type
     * @param SharpFieldFormatter|null $formatter
     */
    protected function __construct(string $name)
    {
        $this->name = $name;
    }

     /**
     * @param string $key
     * @return this
     */
    public static function make(string $name)
    {
        return new static($name);
    }

    /**
     * @param string $label
     * @return static
     */
    public function setLabel(string $label)
    {
        $this->label = $label;

        return $this;
    }

    /**
     * @param string $helpMessage
     * @return static
     */
    public function setHelpMessage(string $helpMessage)
    {
        $this->helpMessage = $helpMessage;

        return $this;
    }

    /**
     * @param bool $readOnly
     * @return static
     */
    public function setReadOnly(bool $readOnly = true)
    {
        $this->readOnly = $readOnly;

        return $this;
    }

     /**
     * @param string $database table field name
     * @return static
     */
    public function field(string $fieldName)
    {
        $this->field = $fieldName;

        return $this;
    }

    public function hideFromIndex(){
        
    }
    public function hideFromDetail(){
        
    }
    public function hideWhenCreating(){
        
    }
    public function hideWhenUpdating(){
        
    }
    public function onlyOnIndex(){
        
    }
    public function onlyOnDetail(){
        
    }
    public function onlyOnForms(){
        
    }
    public function exceptOnForms(){
        
    }

    /**
     * Create the properties array for the field, using parent::buildArray()
     *
     * @return array
     */
    public abstract function toArray(): array;


    /**
     * @return string
     */
    public function name(): string
    {
        return $this->name;
    }

    /**
     * Return specific validation rules.
     *
     * @return array
     */
    protected function validationRules()
    {
        return [];
    }

    /**
     * Throw an exception in case of invalid attribute value.
     *
     * @param array $properties
     * @throws SharpFormFieldValidationException
     */
    protected function validate(array $properties)
    {
        $validator = Validator::make($properties, [
            'key' => 'required',
            'type' => 'required',
        ] + $this->validationRules());

        if ($validator->fails()) {
            throw new SharpFormFieldValidationException($validator->errors());
        }
    }

    /**
     * @param array $childArray
     * @return array
     * @throws SharpFormFieldValidationException
     */
    protected function buildArray(array $childArray)
    {
        $array = collect([
            "key" => $this->key,
            "type" => $this->type,
            "label" => $this->label,
            "readOnly" => $this->readOnly,
            "conditionalDisplay" => $this->buildConditionalDisplayArray(),
            "helpMessage" => $this->helpMessage,
            "extraStyle" => $this->extraStyle,
        ] + $childArray)
            ->filter(function($value) {
                return !is_null($value);
            })->all();

        $this->validate($array);

        return $array;
    }
}