<?php
namespace Modules\Cart\Traits;

use Illuminate\Support\Collection;
use Modules\Cart\Entities\CartItem;
use FontLib\Table\Type\post;
use PhpOffice\PhpSpreadsheet\RichText\Run;

trait CartPrice {

    public function postPrice() {
        // status:
        //     0 : without post
        //      1: post OK
        //     400 : unknown post send
        //     100: threshold Run
        //      2: bike postfix  پسکرایه پیک
        //      3: normal postfix پست پسکرایه

        $content = $this->getContent();
        if (!array_key_exists('items', $content)) {
            $content['items'] = new Collection;
        }

        if(!config('cart.post.status')){
            return ['status' => 0, 'price' => 0];
        }

        $total = $content['items']->reduce(function ($carry, CartItem $cartItem) {
            $notCompute = config('cart.post.notCompute');
            $temp = !in_array($cartItem->id, $notCompute) ? ($cartItem->qty * ($cartItem->priceTax - $cartItem->discount)) : 0;
            return $carry + $temp;
        }, 0);
//dd([$t,$total]);
        $count = $this->count();
        $postMethod = $this->getPostMethod();
        if ($count < 1 || $postMethod == 'unknown') {
            return ['status' => 400, 'price' => 0];
        }

        if ($total > config('cart.post.threshold')) {
            return ['status' => 100, 'price' => 0];;
        } else {
            switch ($postMethod) {
                case 'bikePost':
                    return config('cart.post.bikepost.status') ? ['status' => 1, 'price' => config('cart.post.bikepost.price')] : ['status' => 2, 'price' => 0];
                    break;
                
                case 'normalpost':
                default:
                    $price = config('cart.post.normalpost.base') + (config('cart.post.normalpost.per') * ($count - 1));
                    return config('cart.post.normalpost.status') ? ['status' => 1, 'price' => $price] : ['status' => 3, 'price' => 0];
                    break;
            }
        }

    }

    /**
     * Get the subtotal (total - tax) of the items in the cart.
     *
     * @param int    $decimals
     * @param string $decimalPoint
     * @param string $thousandSeperator
     * @return float
     */
    public function subtotal() {
        $content = $this->getContent();
        if (!array_key_exists('items', $content)) {
            $content['items'] = new Collection;
        }

        $subTotal = $content['items']->reduce(function ($subTotal, CartItem $cartItem) {
            $temp = ($cartItem->qty * $cartItem->priceTax);
            return $subTotal + $temp;
        }, 0);

        return $subTotal;
    }

    public function calculateOptionPrice() {
        $postPrice = $this->postPrice();
        $detail = [
            'post' => [
                'price' => $postPrice['price'],
                'status' => $postPrice['status'],
                'method' => $this->getPostMethod(),
            ],
        ];
        return [
            'detail' => $detail,
            'total' => $postPrice['price'],
        ];
    }

    /**
     * Get the total price of the items in the cart.
     *
     * @param int    $decimals
     * @param string $decimalPoint
     * @param string $thousandSeperator
     * @return string
     */
    public function totalAll() {

        $discount = $this->checkDiscount();
        $optionPrice = $this->calculateOptionPrice();

        return collect([
            'total' => $discount['subtotal'] + $optionPrice['total'] - $discount['discount'],
            'discount' => $discount['discount'],
            'discountDetail' => $discount['discountDetail'],
            'subtotal' => $discount['subtotal'],
            'post' => $optionPrice['detail']['post']['price'],
            'postStatus' => $optionPrice['detail']['post']['status'],
            'postMethod' => $optionPrice['detail']['post']['method'],
            'count' => $this->count(),
        ]);

    }

    private function getSubTotalDiscount() {

        $content = $this->getContent();
        $discount = 0;
        $subtotal = 0;
        $notComputeTotal = 0;
        $productsId = [];
        if (!array_key_exists('items', $content)) {
            $content['items'] = new Collection;
        }

        $content['items']->each(function ($cartItem) use (&$discount, &$subtotal, &$notComputeTotal, &$productsId) {
            $productsId[] = $cartItem->id;
            $notCompute = config('cart.discount.notCompute');
            $notComputeTotal += in_array($cartItem->id, $notCompute) ? ($cartItem->qty * $cartItem->priceTax) : 0;
            $discount += ($cartItem->qty * $cartItem->discount);
            $subtotal += ($cartItem->qty * $cartItem->priceTax);
        });

        return [
            'subtotal' => $subtotal,
            'discount' => $discount,
            'festivalDiscount' => 0,
            'notComputTotal' => $notComputeTotal,
            'productsId' => $productsId,
        ];
    }

    public function itemTotalAll($rowId) {
        $content = $this->getContent();
        if (!array_key_exists('items', $content)) {
            $content['items'] = new Collection;
        }

        if (!$content['items']->has($rowId)) {
            throw new InvalidRowIDException("The cart does not contain rowId {$rowId}.");
        }

        return $content['items']->get($rowId)->totalAll();
    }

    /**
     * Get the total tax of the items in the cart.
     *
     * @param int    $decimals
     * @param string $decimalPoint
     * @param string $thousandSeperator
     * @return float
     */
    public function tax($decimals = null, $decimalPoint = null, $thousandSeperator = null) {
        $content = $this->getContent();
        if (!array_key_exists('items', $content)) {
            $content['items'] = new Collection;
        }

        $tax = $content['items']->reduce(function ($tax, CartItem $cartItem) {
            return $tax + ($cartItem->qty * $cartItem->tax);
        }, 0);

        return $this->numberFormat($tax, $decimals, $decimalPoint, $thousandSeperator);
    }

    public function totalDiscount($decimals = null, $decimalPoint = '.', $thousandSeperator = ',') {
        $content = $this->getContent();
        if (!array_key_exists('items', $content)) {
            $content['items'] = new Collection;
        }

        $subTotal = $content['items']->reduce(function ($subTotal, CartItem $cartItem) {
            return $subTotal + ($cartItem->qty * $cartItem->discount);
        }, 0);

        return $this->numberFormat($subTotal, $decimals, $decimalPoint, $thousandSeperator);
    }
}