<?php namespace Modules\Book\Models;

use App;
use App\Models\Model;
use DB;
use Modules\Author\Models\Author;
use Modules\Cart\Models\Invoice;
use Modules\Comment\Models\Comment;
use Modules\Festival\Models\Festival;
use Modules\Home\Models\Tag;
use Modules\Publisher\Models\Publisher;

class Book extends Model {
    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'xxbook';

    /**
     * The primary key for the model.
     *
     * @var string
     */
    protected $primaryKey = 'xbookid';

    /**
     * Indicates if the model should be timestamped.
     *
     * @var bool
     */
    public $timestamps = false;

    protected $guarded = ['xbookid'];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [
        'xauthorid' => 'int',
        'xtranslatorid' => 'int',
        'xpublisherid' => 'int',
        'xbook_hard_price' => 'int',
        'xbook_hard_discount' => 'int',
        'xbook_digital_price' => 'int',
        'xbook_year' => 'int',
        'xbook_pages' => 'int',
        'xrank' => 'int',
    ];

    protected $relation = [
        'author',
        'translator',
        'publisher',
    ];

    /**
     * Get the gallery for the book.
     */
    public function images() {
        return $this->hasMany(BookImage::class, 'xbookid');
    }

    /**
     * Get the author that owns the book.
     */
    public function author() {
        return $this->belongsToMany(Author::class, 'xxbook_author', 'xbookid', 'xauthorid');
    }

    /**
     * Get the author that owns the book.
     */
    public function translator() {
        return $this->belongsToMany(Author::class, 'xxbook_translator', 'xbookid', 'xauthorid');
    }

    /**
     * Get the publisher that owns the book.
     */
    public function publisher() {
        return $this->belongsTo(Publisher::class, 'xpublisherid');
    }

    /**
     * Get all of the book's comments.
     */
    public function comments() {
        return $this->morphMany(Comment::class, 'xcommentable');
    }

    /**
     * Get the invoice for the book.
     */
    public function invoices() {
        return $this->belongsToMany(Invoice::class, 'xxshop_invoiceitem', 'xbookid', 'xinvoiceid')->withPivot('xinvoiceitem_price', 'xinvoiceitem_discount', 'xinvoiceitem_quantity');
    }

    /**
     * Get the invoice for the book.
     */
    public function festival() {
        return $this->belongsToMany(Festival::class, 'xxfestival_book_item', 'xbookid', 'xfestivalid')->withPivot('xdiscount');
    }

    /**
     * Get the category that owns the book.
     */
    public function category() {
        return $this->belongsTo(SubCategory::class, 'xsubcategoryid');
    }

    public function favorites() {
        return $this->belongsToMany(User::class, 'xxuser_favorite', 'xbookid', 'xuserid');
    }

    public function tags() {
        return $this->morphToMany(Tag::class, 'taggable', 'xxtaggables', 'taggable_id', 'tag_id');
    }

    public function discountPrice(){
        $perProduct = $this->discount;
        $festival = $this->festival()->orderBy('xrank','desc')->first();
        $festival = $festival ? $festival->pivot->xdiscount : 0;
        return $festival ? $festival : ($perProduct ? $perProduct : 0);
    }

    public static function updateById($data, $id = 0) {
        $instance = new static();
        return $instance->updateOrCreate([$instance->primaryKey => $id], $data);
    }

    /**
     * get Result
     * @param $query
     * @return mixed
     */
    public static function getResult() {
        $instance = new static();

        $res = DB::table($instance->table)
            ->leftJoin('xxbook_category_sub', 'xxbook_category_sub.xsubcategoryid', '=', 'xxbook.xsubcategoryid')
            ->leftJoin('xxbook_category as c', 'xxbook_category_sub.xcategoryid', '=', 'c.xcategoryid')
            ->leftJoin('xxpublisher', 'xxpublisher.xpublisherid', '=', 'xxbook.xpublisherid')
            ->select(['*', "{$instance->primaryKey} AS xid"]);

        return $res;
    }

    /**
     * delete
     * @param $id
     * @return mixed
     */
    public static function remove($id) {
        $instance = new static();

        if ($id) {
            $images = $instance->find($id)->images;

            foreach ($images as $image) {
                parent::deleteFile($image->xbook_imageid, 'gallery');
                $image->delete();
            }

            $res = $instance->destroy($id);
            parent::deleteFile($id);
            die('[[OK]]');
        } else {
            die('[[Error]]');
        }

        return $res;
    }

    /**
     * Hire car models
     * @param $id
     * @return mixed
     */

    public static function subSelect() {
        $sql = "SELECT *
                    FROM (
                    SELECT xbookgroupid, GROUP_CONCAT( xbooksubgroupid,  ':', xbooksubgroup_title ) AS subselect
                    , GROUP_CONCAT( xbooksubgroupid,  ':', xbooksubgroup_title_en ) AS subselectEn
                    , GROUP_CONCAT( xbooksubgroupid,  ':', xbooksubgroup_title_ar ) AS subselectAr
                    FROM  `xxbooksubgroup`
                    GROUP BY xbookgroupid
                    )q1
                    LEFT JOIN xxbookgroup
                    USING ( xbookgroupid )
                    where xbookgroup_lang = '" . App::getLocale() . "'";
        $res = DB::select($sql);

        return $res;
    }

    public static function price($color) {
        if ($color) {
            return this;
        }

    }
}