<?php

namespace Modules\Blog\Http\Controllers\Index;

use App\Http\Controllers\IndexController;
use FarsiLib;
use File;
use Modules\Blog\Models\Article;
use Modules\Blog\Models\RatingInfo;
use Auth;
use Modules\Blog\Models\ArticleGroup;
use Request;
use Modules\Home\Models\Tag;
use Modules\Advertise\Models\Advertise;
use Input;
use Modules\Blog\Entities\SeoSchema;

class BlogController extends IndexController
{

    /*
    |--------------------------------------------------------------------------
    | Home Controller
    |--------------------------------------------------------------------------
    |
    | This controller renders your application's "dashboard" for users that
    | are authenticated. Of course, you are free to change or remove the
    | controller as you wish. It is just here to get your app started!
    |
     */

    /**
     * Show the application dashboard to the user.
     *
     * @return Response
     */
    public function index($title)
    {
        if ($title) {
            $title = str_replace('-', ' ', $title);
            $id = Article::where('url', $title)->first();
          
            $id = $id ? $id->xarticleid : null;
            if ($id) {
                return $this->detailArticle($id);
            }
        }

        $article = Article::with(['user', 'group'])
            ->where('xlang', 'fa')
            ->where('xstatus', 1)
            ->orderBy('updated_at', 'desc')
            ->paginate(10)->onEachSide(3)->withPath(Request::fullUrl());

        $articleGrp = ArticleGroup::getresult()
            ->orderBy('xrank', 'desc')
            ->get();

        $topArticles = Article::where('xarticle_top', '1')
            ->where('xlang', 'fa')
            ->where('xstatus', 1)
            ->orderBy('xrank', 'desc')
            ->orderBy('created_at', 'desc')
            ->getWithLocale();

        $selected = Article::where('xarticlegroupid', '8')
            ->where('xlang', 'fa')
            ->where('xstatus', 1)
            ->orderBy('xrank', 'desc')
            ->orderBy('created_at', 'desc')
            ->getWithLocale();

        // $this->layout->breadcrumb = [['title' => 'blog', 'href' => '']];

        $this->layout->title2 = t('blog');
        $this->layout->content = view('blog::index.article', [
            'article' => $article,
            'articleGrp' => $articleGrp,
            'topArticles' => $topArticles,
            'selected' => $selected,
        ]);
    }

    public function detail($id)
    {
        if ($id) {
            $article = Article::find($id);
            if (!$article) {
                if (config('app.theme') == 'goldiran') {
                    return redirect('/blog/category/3');
                } else
                    return redirect('/blog');
            }
            $title = str_replace(' ', '-', $article->url);
            return redirect("/" . \App::getLocale() . "/blog/$title", 301);
        }
    }

    public function detailArticle($id)
    {
        $article = Article::withCount([
            'rating as like_count' => function ($query) {
                $query->where('action', 1);
            },
            'rating as dislike_count' => function ($query) {
                $query->where('action', 2);
            },
            'comments' => function ($query) {
                $query->where('xcomment_approved', 1);
            }
        ])->find($id);
        if (!$article) return redirect('/');
        if($article->redirect_url)  return redirect($article->redirect_url,301);
        $this->layout->title2 = $article->xarticle_title;
        $bannerExist = File::exists(config('upload.' . config('app.controller')) . "/banner/$id");
        $comments = $article->comments()
            ->where('xcomment_approved', 1)
            ->where('xparentid', 0)
            ->orderBy('xparentid', 'asc')
            ->orderBy('xcomment_date', 'desc')
            ->get();

        $advertise = Advertise::leftJoin('xxadvertise_place', function ($join) {
            $join->on('xxadvertise.xplaceid', '=', 'xxadvertise_place.xplaceid');
        })->where('xlocation', 'Blog')
            ->where('xlocationid', $article->xarticlegroupid)
            ->where('xxadvertise.xstatus', 1)
            ->orderBy('xxadvertise_place.xrank', 'desc')
            ->orderBy('xxadvertise.xrank', 'desc')
            ->get();

        $this->layout->content = view('blog::index.article_detail', [
            'article' => $article,
            'bannerExist' => $bannerExist,
            'comments' => $comments,
            'advertise' => $advertise,
        ]);

        $this->layout->description = $article->seo_description;
        $this->layout->title2 = $article->seo_title;
        $this->layout->canonical = $article->seo_canonical;
        $this->layout->seoNoIndex = $article->seo_noindex;

        $this->layout->schemaArticle = json_encode(SeoSchema::schemaArticle($article, $this->config['seo']['title']), JSON_UNESCAPED_SLASHES);
        $this->layout->schemaBreadCrump = json_encode(SeoSchema::schemaBreadcrumb($article), JSON_UNESCAPED_SLASHES);
        $this->layout->schemaProduct = json_encode(SeoSchema::schemaProductForArticle($article), JSON_UNESCAPED_SLASHES);

        $this->layout->content->breadcrumb = [
            ['title' => $article->group->xarticlegroup_title, 'href' => 'Blog/category/' . $article->group->xarticlegroupid],
            ['title' => FarsiLib::faNormalize($article->xarticle_title, true), 'href' => ''],
        ];
    }

    private function depthSearch($node, $list, &$newList)
    {
        $newList[$node->xcommentid] = ['node' => $node, 'children' => false];
        $childs = $list->filter(function ($item) use ($node) {
            return $item->xparentid == $node->xcommentid;
        });
        foreach ($childs as $child) {
            self::depthSearch($child, $list, $newList[$node->xcommentid]['children']);
        }
    }

    public function search()
    {
        $str = Input::get('src');
        if (!$str)
            $items = Article::orderBy('created_at', 'desc')->where('xlang', 'fa')->where('xstatus', 1)->paginate(7)->withPath(Request::fullUrl());
        else
            $items = Article::where('xarticle_title', 'like', "%{$str}%")
                ->orWhere('xarticle_desc', 'like', "%{$str}%")
                ->orderBy('created_at', 'desc')
                ->where('xlang', 'fa')
                ->where('xstatus', 1)
                ->paginate(7)->withPath(Request::fullUrl());

        $articleGrp = ArticleGroup::orderBy('xrank', 'desc')->getWithLocale();
        //$products = Product::where('show_repaire', 1)->getWithStatus()->prepend(new Product(['id' => 0, 'name' => 'همه محصولات']));
        $bestArticle = Article::where('xstatus', 1)->orderBy('xrank', 'desc')->take(5)->getWithLocale();

        $this->layout->content = view('blog::index.articleCategory', [
            'bestArticle' => $bestArticle,
            'list' => $items,
            'type' => 'blog',
            'groups' => $articleGrp,
            'mygroup' => $articleGrp[0],
            //'products' => $products,
        ]);
    }

    public function category($id)
    {
        $brand = Input::get('brand');
        //$product = Input::get('product');
        if (!$id) {
            $id = ArticleGroup::first()->xarticlegroupid;
        }
        $category = ArticleGroup::where('xarticlegroupid',$id)->first();
        $articleGrp = ArticleGroup::where('parent_id',$category->parent_id ? $category->parent_id : $category->xarticlegroupid)->orderBy('xrank', 'desc')->getWithLocale();

        $article = Article::whereIn('xarticlegroupid', $category->parent_id ? [$id] : $articleGrp->pluck('xarticlegroupid'))
            ->where('xstatus', 1)
            ->orderBy('created_at', 'desc')
            ->when($brand, function ($query, $data) {
                $tag = Tag::where('xname', $data)->first();
                $query->whereIn('xarticleid', $tag ? $tag->articles->pluck('xarticleid') : [0]);
            })
            // ->when($product, function ($query, $data) {
            //     if ($data != 'همه محصولات') {
            //         $tag = Tag::where('xname', $data)->first();
            //         $query->whereIn('xarticleid', $tag ? $tag->articles->pluck('xarticleid') : [0]);
            //     }
            // })
            ->paginate(10)->onEachSide(3)->withPath(Request::fullUrl());

        $bestArticle = Article::whereIn('xarticlegroupid',$category->parent_id ? [$category->xarticlegroupid] : $articleGrp->pluck('xarticlegroupid'))->where('xstatus', 1)->orderBy('xrank', 'desc')->take(8)->getWithLocale();
        $group = $category;
        $breadcrumb[] = ['title' => t('blog'), 'href' => '/Blog'];
        $breadcrumb[] = ['title' => $group->xarticlegroup_title, 'href' => ''];
        //$this->layout->breadcrumb = $breadcrumb;
        //$products = Product::where('show_repaire', 1)->getWithStatus()->prepend(new Product(['id' => 0, 'name' => 'همه محصولات']));
        $this->layout->title = $group->xarticlegroup_title;
        //$productFilter = $products->where('name', '%like%', $product)->first();

        $this->layout->canonical = config('app.url') . '/blog/category/';
        $this->layout->content = view('blog::index.articleCategory', [
            'bestArticle' => $bestArticle,
            'list' => $article,
            'groups' => $articleGrp,
            'mygroup' => $group,
            //'products' => $products,
            //'activeGroup' => $product ? ($productFilter ? $productFilter->name : null) : null,
            'activeBrand' => $brand
        ]);
    }

    public function addRating()
    {
        $type = Input::get('type');
        $id = Input::get('articleid');
        $user = Auth::user();

        if (!$user) {
            return response()->json(['success' => false, 'message' => 'لطفا در سیستم وارد شوید', 'type' => 0], 401);
        }

        $res = RatingInfo::updateById(['action' => $type], ['user_id' => $user->xuserid, 'article_id' => $id]);
        $article = Article::withCount([
            'rating as like_count' => function ($query) {
                $query->where('action', 1);
            },
            'rating as dislike_count' => function ($query) {
                $query->where('action', 2);
            }
        ])->find($id);
        return response()->json(['success' => true, 'like' => $article->like_count, 'dislike' => $article->dislike_count, 'message' => 'عملیات با موفقیت انجام شد', 'type' => 100]);
    }
}
