<?php

namespace Modules\Bank\Entities\Fanava;

use Illuminate\Support\Facades\Input;
use Modules\Bank\Entities\PortAbstract;
use Modules\Bank\Entities\PortInterface;
use SoapClient;

class Fanava extends PortAbstract implements PortInterface {
    protected $tokenUrl = 'https://fanava.shaparak.ir/nafis/token_manager/AuthenticateUser/';
    /**
     * Url of fanava gateway web service
     *
     * @var string
     */
    protected $serverUrl = 'https://fanava.shaparak.ir/nafis/_ipgn_';

    /**
     * Address of gate for redirect
     *
     * @var string
     */
    protected $confirmUrl = 'https://fanava.shaparak.ir/nafis/extsrv-ipg/Confirmation.svc?wsdl';

    /**
     * {@inheritdoc}
     */
    public function set($amount) {
        $this->amount = intval($amount);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function ready() {
        $this->sendPayRequest();

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function redirect() {
        return \View::make('bank::index.fanava-redirector')->with([
            'token' => $this->refId,
            'terminalCode' => $this->config->get('bank.fanava.terminalId'),
            'merchantCode' => $this->config->get('bank.fanava.merchantId'),
            'terminalSerial' => $this->config->get('bank.fanava.terminalSerial'),
            'amount' => $this->amount,
            'invoiceNumber' => $this->invoiceId,
            'stan' => substr($this->transactionId(), 6, 12),
            'ip' => '91.98.96.132',
            'redirectURL' => $this->getCallback(),
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public function verify($transaction) {
        parent::verify($transaction);

        $this->verifyPayment();

        return $this;
    }

    /**
     * Sets callback url
     * @param $url
     */
    function setCallback($url) {
        $this->callbackUrl = $url;
        return $this;
    }

    /**
     * Gets callback url
     * @return string
     */
    function getCallback() {
        if (!$this->callbackUrl) {
            $this->callbackUrl = $this->config->get('bank.fanava.callback-url');
        }

        return $this->makeCallback($this->callbackUrl, ['transaction_id' => $this->transactionId()]);
    }

    protected function getToken() {
        $params = [
            'Username' => $this->config->get('bank.fanava.username'),
            'Password' => $this->config->get('bank.fanava.password'),
        ];

        $ch = curl_init();

        curl_setopt($ch, CURLOPT_URL, $this->tokenUrl);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

        $response = curl_exec($ch);
        $response = json_decode($response, true);
        curl_close($ch);

        if (is_numeric($response['ResponseCode']) && $response['ResponseCode'] == '0000') {
            $this->refId = $response['Token'];
            $this->transactionSetRefId();
            return true;
        }

        $this->transactionFailed();
        $this->newLog(-1, 'خطا در اتصال به درگاه پارسیان');
        throw new FanavaErrorException('خطا در اتصال به درگاه پارسیان', -1);

    }

    /**
     * Send pay request to parsian gateway
     *
     * @return bool
     *
     * @throws ParsianErrorException
     */
    protected function sendPayRequest() {
        $this->newTransaction();
        $this->getToken();
    }

    /**
     * Verify payment
     *
     * @throws ParsianErrorException
     */
    protected function verifyPayment() {

        $reqType = Input::get('BuyModel_RequestType');
        $reqCode = Input::get('BuyModel_ResponseCode');
        $msg = Input::get('BuyModel_ResponseMsg');
        $amount = Input::get('BuyModel_Amount');
        $refNo = Input::get('BuyModel_RefNo');
        $reqDate = Input::get('RequestDate');
        $reqTime = Input::get('RequestTime');
        $status = Input::get('Status');
        $resCode = Input::get('RESCODE');
        $this->cardNumber = Input::get('Pan');

        if ($status != 'Ok') {
            $this->transactionFailed();
            $this->newLog($resCode, $msg);
            throw new FanavaErrorException($resCode, $this->invoiceId);
        }
        if ((int) $amount != (int) $this->amount) {
            $this->transactionFailed();
            $this->newLog($reqCode, 'amount not equial - ' . $msg);
            throw new FanavaErrorException('تراکنشی یافت نشد', $reqCode);
        }

        $SettlementData = new \stdClass();
        $SettlementData->Amount = $this->amount;
        $SettlementData->MerchantCode = $this->config->get('bank.fanava.merchantId');
        $SettlementData->RefNo = $refNo;
        $SettlementData->RequestDate = $reqDate;
        $SettlementData->RequestTime = $reqTime;
        $SettlementData->RequestType = $reqType;
        $SettlementData->Stan = substr($this->transactionId(), 6, 12);
        $SettlementData->TerminalCode = $this->config->get('bank.fanava.terminalId');
        $SettlementData->TerminalSerial = $this->config->get('bank.fanava.terminalSerial');
        $SettlementData->Token = Input::get('Token');

        try {
            $soap = new SoapClient($this->confirmUrl);
            $result = $soap->Settlement(array('input' => $SettlementData));
        } catch (\SoapFault $e) {
            throw new FanavaErrorException($e->getMessage(), -1);
        }

        if ($result->SettlementResult->ResponseCode != '0' || $result->SettlementResult->ResponseCode != '00' || $result->SettlementResult->ResponseCode != '0000') {
            $errorMessage = FanavaResult::errorMessage($result);
            $this->transactionFailed();
            $this->newLog($result, $errorMessage);
            throw new FanavaErrorException($errorMessage, $result);
        }

        $this->trackingCode = $result->SettlementResult->RefNo;
        $this->transactionSucceed();
        $this->newLog($result->SettlementResult->ResponseCode, $result->SettlementResult->RefNo);

    }
}
